// I2CBus.cpp: implementation of the I2CBus class.
//
// code ported from the visual basic source code of the program 
// "I2CTools" and "ServoCtr" by OOPic.
// Alex Barth Dec. 2002
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "I2CBus.h"
#include <conio.h> 
#include <stdio.h>

#define LPTPORT	                   0x378 /* "&H378" */
#define OOPIC_PROGRAMMINGDELAY     793   /*  150    */
#define I2C_ICNORECLSTATUS         false /* see OOPic/Hardware/IgnoreCLStatus state in win registry */
#define I2C_DEBUG				   0


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

I2CBus::I2CBus()
{
	m_szLastMessage = new char[2048];
	
	// fixed LPT port
	ResolveLPTPort( LPTPORT );
}

I2CBus::~I2CBus()
{
	delete[] m_szLastMessage;
}


/*******************************************************************************/
/*******************************************************************************/
/**                                                                           **/
/**  I2C Routines                                                             **/
/**                                                                           **/
/*******************************************************************************/
/*******************************************************************************/


bool I2CBus::I2CSendStart()
{
	/*******************************************************************************/
	/* This routine sends a Start command to the I2C Port.                         */
	/* --------------------------------------------------------------------------- */
	/* A Start Command is used to tell all I2C devices to pay attention to the     */
	/* next I2C byte that is being sent, which will be the I2C address byte.       */
	/* Each device on the I2C network will compare its address to that I2C address */
	/* byte, and if it matches, that I2C device will become active, while all the  */
	/* others will become dormant until an Stop Command is sent.                   */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The Start Command is done by changing the I2C-Data from High-to-Low        */
	/*  while the I2C-Clock is High.                                               */
	/* Before:                                                                     */
	/*  The current state of the I2C-Data and the I2C-Clock should be High, Just   */
	/*  in case they are not, this routine sets them.                              */
	/* After:                                                                      */
	/*  The I2C-Date will be Low and the I2C-Clock will be High.                   */
	/*  Once the Start Command has been sent, set the I2C-Clock to Low to prepare  */
	/*  for the next operation.                                                    */
	/*******************************************************************************/

	/*Just in case, set the I2C-Data & I2C-Clock High.*/
	I2CSetDataLine(1);  
	I2CSetClockLine(1);

	if ( I2C_ICNORECLSTATUS == false )
	{
		if ( !I2CReadClock() )
		{
			ClockGroundedMessage();
			return false;
		}
	}

	/*While the I2C-Clock is High, set the I2C-Data Low.*/
	I2CSetDataLine(0);

	if ( I2CReadData())
	{
		DataNotGroundedMessage();
		return false;
	}

	/*Now that the Start signal is done, set the I2C-Clock Low so that the
	  I2C-Data can be changed later without it looking like a stop command.*/
	I2CSetClockLine(0);

	if ( I2C_ICNORECLSTATUS == false )
	{
		if ( I2CReadClock() )
		{
			ClockNotGroundedMessage();
			return false;
		}
	}

	return true;
}


void I2CBus::I2CSendStop()
{
	/*******************************************************************************/
	/* This routine sends a Stop command to the I2C Port.                          */
	/* --------------------------------------------------------------------------- */
	/* A Stop Command is used to tell all I2C devices that the current             */
	/* communications packet has finished. In response to this, they all start     */
	/* waiting for the next Start Command to be sent.                              */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The Stop is done by changing the I2C-Data line from Low-to-High while the  */
	/*  I2C-Clock is High.                                                         */
	/* Before:                                                                     */
	/*  The current state of the I2C-Data needs to be Low and the current state of */
	/*  I2C-Clock needs to be High. Just in case they are not, this routine sets   */
	/*  them.                                                                      */
	/* After:                                                                      */
	/*  The I2C-Clock and I2C-Data will be to be High.                             */
	/*******************************************************************************/

	/*Just in case they are not set, set the I2C-Data Low & I2C-Clock High.*/
	I2CSetDataLine(0);
	I2CSetClockLine(1);
	/*While the I2C-Clock is High, set the I2C-Data High.*/
	I2CSetDataLine(1); /* Set I2C-Data high */
  
}


void I2CBus::I2CSendAck()
{
/*******************************************************************************/
/* This routine sends an Ack to the I2C device.                                */
/* --------------------------------------------------------------------------- */
/* After reading 8 bits out of an I2C device, an Ack must be sent to tell the  */
/* I2C device that more data will be read.                                     */
/*******************************************************************************/
/* How:                                                                        */
/*  The Ack is done by setting the I2C-Data Low, then doing a clock cycle.     */
/* Before:                                                                     */
/*  The current state of the I2C-Clock needs to be Low                         */
/* After:                                                                      */
/*  The I2C-Data must be High because;                                         */
/*  if we are reading data from an I2C device, when the next I2C-Clock strobe  */
/*  happens, we need to be able to read the device's data.                     */
/*  if we are writing data to an I2C device, then the I2C-Data will be set by  */
/*  the data-send routine before the I2C-Clock.                                */
/*  Either way, we can set the I2C-Data High here, and it will work.           */
/*******************************************************************************/
  
	/* Set the I2C-Data Low to signal the Ack */
	I2CSetDataLine(0);
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);
	/* Release the I2C-Data for future reads. */
	I2CSetDataLine(1);

}


void I2CBus::I2CSendNAck()
{
	/*******************************************************************************/
	/* This routine sends a NAck to the I2C device.                                */
	/* --------------------------------------------------------------------------- */
	/* After reading 8 bits out of an I2C device, a NAck must be sent to tell the  */
	/* I2C device that NO more data will be read.                                  */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The NAck is done by setting the I2C-Data High, then doing a clock cycle.   */
	/* Before:                                                                     */
	/*  The current state of the I2C-Clock needs to be Low                         */
	/* After:                                                                      */
	/*  The I2C-Data must be High because;                                         */
	/*  if we are reading data from an I2C device, when the next I2C-Clock strobe  */
	/*  happens, we need to be able to read the device/s data.                     */
	/*  if we are writing data to an I2C device, then the I2C-Data will be set by  */
	/*  the data-send routine before the I2C-Clock.                                */
	/*  Either way, we can set the I2C-Data High here, and it will work.           */
	/*******************************************************************************/

	/* Set the I2C-Data High to signal the NAck */
	I2CSetDataLine(1);
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);
}


void I2CBus::I2CFlushBitStream()
{
	/*******************************************************************************/
	/* This routine flushes any current data communications packet.                */
	/* --------------------------------------------------------------------------- */
	/* if an I2C communications packet is interrupted, it is possible that the     */
	/* active I2C device will still be expecting to output its data to the I2C Bus */
	/* In this case, the data will need to be drawn out of the device before any   */
	/* new communications packet is started.                                       */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The data can be drawn out of the active I2C device by strobing the         */
	/*  I2CClock 10 times while holding the I2CData High followed by a Start/Stop. */
	/* Before:                                                                     */
	/*  Any unknown I2C state                                                      */
	/* After:                                                                      */
	/*  The I2C-Clock and I2C-Data need to be High.                                */
	/*******************************************************************************/

	/* Set the I2C-Data high so that the 9th bit I2C-Clock strobe will look like a
	   NAck to the I2C device. */
	I2CSetDataLine(1);
	/* Prepare the I2C-Clock for strobing.*/
	I2CSetClockLine(1);
	/* Strobe the I2C-Clock 10 times.*/
	for (int i = 0; i < 10; i++)
	{
		I2CSetClockLine(0);
		I2CSetClockLine(1);
	}

	/* Finish up be sending a Start/Stop command.*/
	I2CSetDataLine(0);
	I2CSetDataLine(1);

}


BYTE I2CBus::I2CReadByte()
{
	/*******************************************************************************/
	/* This routine reads 8 bits out of the active I2C device.                     */
	/* --------------------------------------------------------------------------- */
	/* When an I2C device is addressed for read, the device will return 8 bits of  */
	/* data.                                                                       */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The I2C data is read out of the I2C device by accumulating each of the 8   */
	/*  bits of the byte one at a time with an I2C-Clock strobe between each one.  */
	/* Before:                                                                     */
	/*  The I2C-Clock should be Low.                                               */
	/* After:                                                                      */
	/*  The I2C-Clock needs to be Low.                                             */
	/*******************************************************************************/
	
	BYTE ReadBits;

	/* Set I2C-Clock high */
	I2CSetClockLine(1);

	ReadBits = 0;

	/* Read the first bit of the data stream (X.......) */
	ReadBits = ReadBits + (I2CReadData() * 128);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the second bit of the data stream (.X......) */
	ReadBits = ReadBits + (I2CReadData() * 64);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the third bit of the data stream (..X.....) */
	ReadBits = ReadBits + (I2CReadData() * 32);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the fourth bit of the data stream (...X....) */
	ReadBits = ReadBits + (I2CReadData() * 16);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the fifth bit of the data stream (....X...) */
	ReadBits = ReadBits + (I2CReadData() * 8);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the sixth bit of the data stream (.....X..) */
	ReadBits = ReadBits + (I2CReadData() * 4);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the seventh bit of the data stream (......X.) */
	ReadBits = ReadBits + (I2CReadData() * 2);

	/* Do a I2C-Clock strobe. */
	I2CSetClockLine(0);
	I2CSetClockLine(1);
	/* Read the eighth bit of the data stream (.......X) */
	ReadBits = ReadBits + (I2CReadData() * 1);

	/* Set I2C-Clock low to prepare for the next operation */
	I2CSetClockLine(0);
 
	/* Return the accumulated data bits. */
	return ReadBits;

}


void I2CBus::I2CSendByte( BYTE ByteToSend, BYTE* Ack )
{
	/*******************************************************************************/
	/* This routine writes 8 bits to the active I2C device.                        */
	/* --------------------------------------------------------------------------- */
	/* When an I2C device is addressed for write, the device will store 8 bits of  */
	/* data the is sent to it.                                                     */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The I2C data is written to the I2C device by sending each of the 8 bits of */
	/*  the byte one at a time with a I2C-Clock strobe between each one.  Once All */
	/*  8 bits are sent, the I2C device will send back an Ack or a NAck.           */
	/*  if no device was there to accept the data, a NAck is read.                 */
	/* Before:                                                                     */
	/*  The I2C-Clock should be Low.                                               */
	/* After:                                                                      */
	/*  The I2C-Clock needs to be Low.                                             */
	/*******************************************************************************/

	/* Put bit 7 on the data line */
	I2CSetDataLine(ByteToSend & 128); //0x80  10000000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 6 on the data line */
	I2CSetDataLine(ByteToSend & 64);  //0x40  01000000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 5 on the data line */
	I2CSetDataLine(ByteToSend & 32);  //0x20  00100000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 4 on the data line */
	I2CSetDataLine(ByteToSend & 16);  //0x10  00010000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 3 on the data line */
	I2CSetDataLine(ByteToSend & 8);   //0x08  00001000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 2 on the data line */
	I2CSetDataLine(ByteToSend & 4);   //0x04  00000100
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 1 on the data line */
	I2CSetDataLine(ByteToSend & 2);   //0x02  00000010
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 0 on the data line */
	I2CSetDataLine(ByteToSend & 1);   //0x01  00000001

	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Be sure the I2C-Data is set High so that we can see the Ack or NAck */
	/* coming from the I2C device */
	I2CSetDataLine(1);

	/* Strobe the I2C-Clock and read the Ack/NAck coming from the I2C device. */
	I2CSetClockLine(1);
	if ( I2CReadClock() == 0 )
	{
		// error
	}

	*Ack = I2CReadData();
	
	I2CSetClockLine(0);
}


void I2CBus::I2CSendByte( BYTE ByteToSend )
{
	/*******************************************************************************/
	/* This routine writes 8 bits to the active I2C device.                        */
	/* --------------------------------------------------------------------------- */
	/* When an I2C device is addressed for write, the device will store 8 bits of  */
	/* data the is sent to it.                                                     */
	/*******************************************************************************/
	/* How:                                                                        */
	/*  The I2C data is written to the I2C device by sending each of the 8 bits of */
	/*  the byte one at a time with a I2C-Clock strobe between each one.  Once All */
	/*  8 bits are sent, the I2C device will send back an Ack or a NAck.           */
	/*  if no device was there to accept the data, a NAck is read.                 */
	/* Before:                                                                     */
	/*  The I2C-Clock should be Low.                                               */
	/* After:                                                                      */
	/*  The I2C-Clock needs to be Low.                                             */
	/*******************************************************************************/

	/* Put bit 7 on the data line */
	I2CSetDataLine(ByteToSend & 128); //0x80  10000000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 6 on the data line */
	I2CSetDataLine(ByteToSend & 64);  //0x40  01000000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 5 on the data line */
	I2CSetDataLine(ByteToSend & 32);  //0x20  00100000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 4 on the data line */
	I2CSetDataLine(ByteToSend & 16);  //0x10  00010000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 3 on the data line */
	I2CSetDataLine(ByteToSend & 8);   //0x08  00001000
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 2 on the data line */
	I2CSetDataLine(ByteToSend & 4);   //0x04  00000100
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 1 on the data line */
	I2CSetDataLine(ByteToSend & 2);   //0x02  00000010
	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Put bit 0 on the data line */
	I2CSetDataLine(ByteToSend & 1);   //0x01  00000001

	/* Strobe the I2C-Clock */
	I2CSetClockLine(1);
	I2CSetClockLine(0);

	/* Be sure the I2C-Data is set High so that we can see the Ack or NAck */
	/* coming from the I2C device */
	I2CSetDataLine(1);

	/* Strobe the I2C-Clock and read the Ack/NAck coming from the I2C device. */
	I2CSetClockLine(1);
	if ( I2CReadClock() == 0 ){}

	
	I2CReadData();
	
	I2CSetClockLine(0);
}


void I2CBus::I2CSendAddressW(BYTE I2CAddress, BYTE* Ack)
{
	/*******************************************************************************/
	/* This routine takes a 7 bit address, merges it with a 0 in the Read/Write    */
	/* bit location to indicate a write command, and then sends it to the          */
	/* I2CSendByte routine                                                         */
	/*******************************************************************************/
	I2CSendByte( (I2CAddress * 2) | 0, Ack);
} 



void I2CBus::I2CSendAddressR(BYTE I2CAddress, BYTE* Ack)
{
	/*******************************************************************************/
	/* This routine takes a 7 bit address, merges it with a 1 in the Read/Write    */
	/* bit location to indicate a read command, and then sends it to the           */
	/* I2CSendByte routine                                                         */
	/*******************************************************************************/
	I2CSendByte((I2CAddress * 2) | 1, Ack);
}



void I2CBus::I2CSetClockLine(BYTE LogicLevel)
{
	/*******************************************************************************/
	/* This routine sets the I2C-Clock Line to a Logic-High or Logic-Low state     */
	/* --------------------------------------------------------------------------- */
	/* if a 0 is passed, then the I2C-Clock Line is set to 0 Volts.                */
	/* if anything > than 0 is passed, then the I2C-Clock Line is set to +5 Volts. */
	/* In either case,                                                             */
	/*   Wait for the new Logic-Level to overcome the PC-I2C Cable's Capacitance.  */
	/*******************************************************************************/

	if ( LogicLevel == 0 )
		PortOut( cClockOutPort, cSCLoLow );
	else
		PortOut( cClockOutPort, cSCLoHigh );
	
	I2CDelay();
}



bool I2CBus::I2CReadClock()
{
	/*******************************************************************************/
	/* This routine reads the I2C-Clock Line's logic state                         */
	/* --------------------------------------------------------------------------- */
	/* if the I2C-Clock Line is at 0 Volts, then a 0 is returned                   */
	/* if the I2C-Clock Line is at +5 Volts, then a 1 is returned                  */
	/*******************************************************************************/
	
	if ( ( PortInp(cClockInPort) & cSCLMask) == cSCLiHigh )
		return true;

	return false;
}



void I2CBus::I2CSetDataLine( BYTE LogicLevel )
{
	/*******************************************************************************/
	/* This routine sets the I2C-Data Line to a Logic-High or Logic-Low state      */
	/* --------------------------------------------------------------------------- */
	/* if a 0 is passed, then the I2C-Data Line is set to 0 Volts.                 */
	/* if anything > than 0 is passed, then the I2C-Data Line is set to +5 Volts.  */
	/* In either case,                                                             */
	/*   Wait for the new Logic-Level to overcome the PC-I2C Cable's Capacitance.  */
	/*******************************************************************************/
  
	if ( LogicLevel == 0 )
		PortOut ( cDataOutPort, cSDAoLow );
	else
		PortOut ( cDataOutPort, cSDAoHigh);
	
	I2CDelay();
}


bool I2CBus::I2CReadData()
{
	/*******************************************************************************/
	/* This routine reads the I2C-Data Line's logic state                          */
	/* --------------------------------------------------------------------------- */
	/* if the I2C-Data Line is at 0 Volts, then a 0 is returned                    */
	/* if the I2C-Data Line is at +5 Volts, then a 1 is returned                   */
	/*******************************************************************************/

	if ( ( PortInp(cDataInPort) & cSDAMask) == cSDAiHigh) 
		return true;

	return false;

}



void I2CBus::I2CSetReset(BYTE LogicLevel)
{
	/*******************************************************************************/
	/* This routine sets the I2C-Reset Line to a Logic-High or Logic-Low state     */
	/* --------------------------------------------------------------------------- */
	/* if a 0 is passed, then the I2C-Reset Line is set to 0 Volts.                */
	/* if anything > than 0 is passed, then the I2C-Reset Line is set to +5 Volts. */
	/* In either case,                                                             */
	/*   Wait for the new Logic-Level to overcome the PC-I2C Cable's Capacitance.  */
	/*******************************************************************************/
  
	if (LogicLevel == 0 )
	{
		PortOut(cResetPort, cResetLow);
		cSCLoHigh = 0 + cResetLow;
		cSCLoLow  = 8 + cResetLow;
	}
	else
	{
		PortOut(cResetPort, cResetHigh);
		cSCLoHigh = 0 + cResetHigh;
		cSCLoLow  = 8 + cResetHigh;
	}

	I2CDelay();
}




void I2CBus::I2CDelay()
{
	/*******************************************************************************/
	/* This routine stops the program from running for a specified period of time. */
	/* --------------------------------------------------------------------------- */
	/* It is used when a new Logic-Level is set on the parallel port and needs to  */
	/* overcome the PC-I2C Cable's Capacitance.                                    */
	/*******************************************************************************/
	
	double f;

	if (I2C_DEBUG)
		for (f = 0.0; f < gDelayTime * 10 ; f+=0.1);
	else
		for (f = 0.0; f < gDelayTime;       f+=0.1);
	
}


// actual port read function
// simplified. see original function in servoctr.
//
BYTE I2CBus::PortInp(int zPortAddress)
{
	return _inp( (unsigned short) zPortAddress);
}

// actual port send function
// simplified. see original function in servoctr.
//
void I2CBus::PortOut(int zPortAddress, BYTE zValue)
{	
	_outp( (unsigned short)zPortAddress, (int)zValue);
}


void I2CBus::ResolveLPTPort( int PortAddress )
{
	/*******************************************************************************/
	/* This routine sets the port address to use for the specified parallel port   */
	/*******************************************************************************/

	gDelayTime = OOPIC_PROGRAMMINGDELAY;
 
	/* pin 1 - Centronics interface: Reset*/
	cResetPort = PortAddress + 2;
	cResetHigh = 0;
	cResetLow = 1;

	/* pin 17 - Centronics interface: Select Input*/
	cClockOutPort = PortAddress + 2;
	cSCLoHigh = 0;
	cSCLoLow = 8;

	/* pin 9 - Centronics interface: Data Bit 7*/
	cDataOutPort = PortAddress;
	cSDAoHigh = 255;
	cSDAoLow = 127;

	/* pin 15 - Centronics interface: Error*/
	cClockInPort = PortAddress + 1;
	cSCLMask = 8;
	cSCLiLow = 0;
	cSCLiHigh = 8;

	/* pin 11 - Centronics interface: Busy*/
	cDataInPort = PortAddress + 1;
	cSDAMask = 128;
	cSDAiLow = 128;
	cSDAiHigh = 0;
   
}



void I2CBus::ClockGroundedMessage()
{
	sprintf(m_szLastMessage,		"The following problem was detected while establishing I2C communications\n\n"
									"The I2C Bus Clock status line reported a logic state of 0, when a logic\n"
									"state of 1 was expected.  This could be caused by any one of the following;\n\n"
									" A. There was no, or not enough, voltage powering the OOPic.\n"
									" B. The I2C Clock line is shorted to ground on the OOPic or in the Cable.\n"
									" C. The I2C Clock status line is shorted to ground in the Cable.\n"
									" D. Another I2C master is controlling the I2C Bus.\n\n"
			);
}


void I2CBus::ClockNotGroundedMessage()
{
	sprintf(m_szLastMessage,	"The following problem was detected while establishing I2C communications.\n"
								"The I2C Bus Clock status line reported a logic state of 1, when a logic\n"
								"state of 0 was expected.  This could be caused by any one of the following;\n\n"
								" A. The programming port is not configured correctly.\n"
								" B. The programming cable was not attached to the PC.\n"
								" C. The programming cable's I2C Clock line is not properly connected.\n"
								" D. The programming cable's I2C Clock status line is not properly connected.\n"
								" E. The I2C Clock line is shorted to +5 Volts on the OOPic or in the Cable.\n"
								" F. The I2C Clock status line is shorted to +5 Volts in the Cable.\n\n"
					  );
  
}


void I2CBus::DataNotGroundedMessage()
{
	sprintf(m_szLastMessage,	"The following problem was detected while establishing I2C communications.\n\n"
								"The I2C Bus Data status line reported a logic state of 1, when a logic\n"
								"state of 0 was expected.  This could be caused by any one of the following;\n\n"
								" A. The programming port is not configured correctly.\n"
								" B. The programming cable was not attached to the PC.\n"
								" C. The programming cable's I2C Data line is not properly connected.\n"
								" D. The programming cable's I2C Data status line is not properly connected.\n"
								" E. The I2C Data line is shorted to +5 Volts on the OOPic or in the Cable.\n"
								" F. The I2C Data status line is shorted to +5 Volts in the Cable.\n\n"
	);
}



void I2CBus::NackMessage()
{
	sprintf(m_szLastMessage,	"The following problem was detected while downloading to the EEPROM.\n\n"
								"The EEPROM did not acknowledge downloaded data.\n"
								"This could be caused by any one of the following;\n\n"
								" A. The programming cable was not attached to the Prg connector on the OOPic.\n"
								" B. The E0 socket on the OOPic did not have an EEPROM in it.\n"
								" C. The EEPROM in the E0 socket on the OOPic was the wrong type.\n"
								" D. The programming cable is damaged.\n"
								" E. The EEPROM in the E0 socket on the OOPic is damaged.\n\n"
					 );
}

// returns last message
//
void I2CBus::I2CGetLastMessage( char* msg )
{
	strncpy(msg, m_szLastMessage, strlen(m_szLastMessage)+1);
}
